/*
 *
 * fits2bin.c
 *
 * Read TIME column of a FITS file and write binary file of doubles
 * in LITTLE ENDIAN byte order.
 *
 */
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>

#include "fitsio.h"

#define STRLEN 256
#define WRITEBUF 2000

void swapx8(char *ptr);
void swapx8_loop(char *ptr, int npts);


/*
 *
 * MAIN
 *
 */
int main(int argc, char **argv)
{

  fitsfile *fptr;
  double MJDREF, MJDREFI, MJDREFF, TIMEZERO, TIMEZERI, TIMEZERF;
  FILE *outfile=NULL;
  int status=0;
  int hdutype;
  long naxis2, num_read, total_events;
  double nulval = -1.0;
  int anynul;
  int timecolnum;
  char fits_name[STRLEN], errstr[STRLEN], commstr[STRLEN];
  double array[3];

  double toffset=-1.0, write_buf[WRITEBUF];
  double firstMJD=1.0e37, lastMJD=0.0;
  double photon_mjd;
  double photon_met;
  int jj=0;

  if ((argc != 3) && (argc != 4)) {
    fprintf(stderr,"Usage: %s fitsfile outfile [MJD Offset]\nReads TIME column of FITS file and writes binary file of doubles (LITTLE_ENDIAN)\n",argv[0]);
    exit(1);
  }
  strncpy(fits_name,argv[1],STRLEN);
  outfile = fopen(argv[2], "wb");
  if (argc == 4)
    toffset = strtod(argv[3], NULL);

  /* Now open the FITS file */
  status=0;
  fits_open_file(&fptr,fits_name,READONLY,&status);

  /* Move to second HDU */
  fits_movabs_hdu(fptr,2,&hdutype,&status);

  /* Get NAXIS2 (number of rows[i.e. events or bins] in file) */
  fits_read_key(fptr,TLONG,"NAXIS2",&naxis2,commstr,&status);

  if (status) {
    fprintf(stderr,"Error opening or positioning file!\n");
    while (ffgmsg(errstr) != 0) {
      fprintf(stderr,"Message: %s\n",errstr);
      exit(1);
    }
  }
  
  status=0;
  /* Get TIMEZERO and MJDREF */
  fits_read_key(fptr,TDOUBLE,"TIMEZERO",&TIMEZERO,commstr,&status);
  if (status) {
    status=0;
    fits_read_key(fptr,TDOUBLE,"TIMEZERI",&TIMEZERI,commstr,&status);
    fits_read_key(fptr,TDOUBLE,"TIMEZERF",&TIMEZERF,commstr,&status);
    if (status) {
      fprintf(stderr,
	      "Error reading TIMEZERO (even tried TIMEZERI,TIMEZERF)!\n");
      while (ffgmsg(errstr) != 0) {
	fprintf(stderr,"Message: %s\n",errstr);
	exit(1);
      }
    }
    TIMEZERO = TIMEZERI + TIMEZERF;
  }
    
  fits_read_key(fptr,TDOUBLE,"MJDREF",&MJDREF,commstr,&status);
  if (status) {
    status=0;
    fits_read_key(fptr,TDOUBLE,"MJDREFI",&MJDREFI,commstr,&status);
    fits_read_key(fptr,TDOUBLE,"MJDREFF",&MJDREFF,commstr,&status);
    if (status) {
      fprintf(stderr,
	      "Error reading MJDREF (even tried MJDREFI,MJDREFF)!\n");
      while (ffgmsg(errstr) != 0) {
	fprintf(stderr,"Message: %s\n",errstr);
	exit(1);
      }
    }
    MJDREF = MJDREFI + MJDREFF;
  }


  fprintf(stderr,"Found %ld events in FT1 file.\n",naxis2);
  /*  fprintf(stderr,"Got HDU of type %d\n",hdutype); */
  fprintf(stderr,"Using MJDREF %.12f, TIMEZERO %.12f\n",MJDREF,TIMEZERO);


  /* 
   * Main Loop: Read times from FITS file
   */
  num_read = 0;
  total_events = 0;
  jj=0;
  fits_get_colnum(fptr,CASEINSEN,"TIME",&timecolnum,&status);
  printf("Found TIME column number %d\n",timecolnum);
  do {
    /* Read TIME column.  This is all we need for photon data */
    fits_read_col(fptr,TDOUBLE,timecolnum,1+num_read,1,1,&nulval,array,&anynul,&status);

    if (status) {
      fprintf(stderr,"Read error!\n");
      while (ffgmsg(errstr) != 0) {
	fprintf(stderr,"Message: %s\n",errstr);
	exit(2);
      }
    }

    if (num_read > 0) {
      /* if (num_read%10000 == 0) fprintf(stderr,"."); */
      /* Insert photon into phase profile */
      
      photon_met = (*array) + TIMEZERO;

      photon_mjd = photon_met/86400.0 + MJDREF;
      if (photon_mjd < firstMJD)
	firstMJD = photon_mjd;
      if (photon_mjd > lastMJD)
	lastMJD = photon_mjd;
      if (toffset < 0.0)
	toffset = photon_mjd;

      write_buf[jj] = photon_mjd - toffset;
      jj++;
      if (jj==WRITEBUF) {
#if 0
#if BYTE_ORDER == BIG_ENDIAN
	/* Swap into LITTLE_ENDIAN if we are on BIG_ENDIAN machine */
	swapx8_loop((char *)write_buf,jj);
#endif
#endif
	fwrite(write_buf, sizeof(double), jj, outfile);
	jj = 0;
      }

      total_events += 1;

    }

    if (anynul) {
      fprintf(stderr,"Got some NULL values!\n");
    }

    num_read += 1;

  } while (num_read < naxis2);

  fits_close_file(fptr,&status);

  if (status) {
    fprintf(stderr,"Error status!\n");
    while (ffgmsg(errstr) != 0) {
      fprintf(stderr,"Message: %s\n",errstr);
    }
  }

#if 0
#if BYTE_ORDER == BIG_ENDIAN
  /* Swap into LITTLE_ENDIAN if we are on BIG_ENDIAN machine */
  swapx8_loop((char *)write_buf,jj);
#endif
#endif
  fwrite(write_buf, sizeof(double), jj, outfile);
  fclose(outfile);

  printf("Converted %ld points.\n"
	 "\tFirst was %20.15g   Last was %20.15g\n" ,
	 num_read, firstMJD, lastMJD);
  printf("Times are in days since EPOCH %.15f\n",toffset);

  exit(0);
}

void swapx8(char *ptr)
{
  char    tmp;

  tmp     = ptr[7];
  ptr[7]  = ptr[0];
  ptr[0]  = tmp;

  tmp     = ptr[6];
  ptr[6]  = ptr[1];
  ptr[1]  = tmp;

  tmp     = ptr[5];
  ptr[5]  = ptr[2];
  ptr[2]  = tmp;

  tmp     = ptr[4];
  ptr[4]  = ptr[3];
  ptr[3]  = tmp;

  return;
}

void swapx8_loop(char *ptr, int npts)
{
  char tmp;
  int i;

  for (i=0; i<npts; i++) {

    tmp     = ptr[7];
    ptr[7]  = ptr[0];
    ptr[0]  = tmp;

    tmp     = ptr[6];
    ptr[6]  = ptr[1];
    ptr[1]  = tmp;

    tmp     = ptr[5];
    ptr[5]  = ptr[2];
    ptr[2]  = tmp;

    tmp     = ptr[4];
    ptr[4]  = ptr[3];
    ptr[3]  = tmp;

    ptr += 8;
  }

  return;
}
