#!/usr/bin/env python
from os import system
from os.path import splitext
import sys
import astropy.io.fits as pyfits
import time
from presto.psr_utils import rad_to_dms, rad_to_hms, coord_to_string
from optparse import OptionParser

RADTODEG = float("57.295779513082320876798154814105170332405472466564")
DEGTORAD = float("1.7453292519943295769236907684886127134428718885417e-2")

desc = 
"""Read an barycentered FT1 FITS file and generate a .events file containing 
binary double precision MJD event times and a .inf file that describes it. These two 
files form the input for prepfold and other PRESTO codes.

Older versions of this code called the output file .bin, but current PRESTO does
not accept that, so the extension is now .events. The name of this code is 
just historical.
"""
parser = OptionParser(
    usage=" %prog [options] [FT1_BARY_FITS_FILENAME]", description=desc
)
parser.add_option(
    "",
    "--source",
    type="string",
    default=None,
    help="Source name to embed in .inf file.",
)

## Parse arguments
(options, args) = parser.parse_args()
if len(args) != 1:
    parser.error("One (and only one) FT1_BARY_FITS_FILENAME argument required.")

fname = args[0]

# fname should be the barycentered FT1 fits file
print("FT1 file: ", fname)
base = splitext(fname)[0]
# print base

hdulist = pyfits.open(fname)
ft1hdr = hdulist[1].header
ft1dat = hdulist[1].data
if ft1hdr["TIMESYS"] != "TDB":
    print("# !!!!!!!!!!!!!!!!!!!!!!!!!!!!")
    print(
        "# !!!!!!!!! WARNING !!!!!!!!!! TIMESYS is NOT TDB!  You probably want to barycenter your photons!!!"
    )
    print("# !!!!!!!!!!!!!!!!!!!!!!!!!!!!")
    time.sleep(5)
# Collect TIMEZERO and MJDREF
try:
    TIMEZERO = ft1hdr["TIMEZERO"]
except KeyError:
    try:
        TIMEZERO = ft1hdr["TIMEZERI"] + ft1hdr["TIMEZERF"]
    except KeyError:
        TIMEZERO = 0.0
print("# TIMEZERO = ", TIMEZERO)
try:
    MJDREF = ft1hdr["MJDREF"]
except KeyError:
    # Here I have to work around an issue where the MJDREFF key is stored
    # as a string in the header and uses the "1.234D-5" syntax for floats, which
    # is not supported by Python
    MJDREF = ft1hdr["MJDREFI"] + float(ft1hdr["MJDREFF"].replace("D", "E"))
print("# MJDREF = ", MJDREF)

ra_deg = ft1hdr["RA_NOM"]
dec_deg = ft1hdr["DEC_NOM"]
ra_str = coord_to_string(*rad_to_hms(DEGTORAD * ra_deg))
dec_str = coord_to_string(*rad_to_dms(DEGTORAD * dec_deg))

mjds = ft1dat.field("TIME") / 86400.0 + MJDREF + TIMEZERO
epoch_mjd = mjds[0]
duration_days = mjds[-1] - mjds[0]

binwidth = 0.001
nbins = int((duration_days * 86400.0) / binwidth)

print("Source Name : ", end=" ")
if options.source is None:
    objname = input()
else:
    objname = options.source
    print(objname)

observer = "None"

# First make .events file
cmd = "fits2bin %s %s.events 0.0" % (fname, base)
print(cmd)
system(cmd)

# Now write out .inf file
inf = open("%s.inf" % base, "w")
infstr = """ Data file name without suffix          =  %s
 Telescope used                         =  GLAST
 Instrument used                        =  LAT
 Object being observed                  =  %s
 J2000 Right Ascension (hh:mm:ss.ssss)  =  %s
 J2000 Declination     (dd:mm:ss.ssss)  =  %s
 Data observed by                       =  GLAST
 Epoch of observation (MJD)             =  %f
 Barycentered?           (1=yes, 0=no)  =  1
 Number of bins in the time series      =  %d
 Width of each time series bin (sec)    =  %f
 Any breaks in the data? (1=yes, 0=no)  =  0
 Type of observation (EM band)          =  Gamma
 Field-of-view diameter (arcsec)        =  3600.00
 Central energy (kev)                   =  300000.0
 Energy bandpass (kev)                  =  100000.0
 Data analyzed by                       =  %s
 Any additional notes:
    None
""" % (
    base,
    objname,
    ra_str,
    dec_str,
    epoch_mjd,
    nbins,
    binwidth,
    observer,
)

inf.write(infstr)
inf.close()
